/* 
   twinklyLayoutRenderer.js
   Unified 3D/2D layout renderer for Twinkly devices.
   Supports rotation, zoom, and live data overlays.
*/

window.TwinklyLayoutRenderer = class TwinklyLayoutRenderer {
    constructor(canvas, container) {
        this.canvas = canvas;
        this.container = container;
        this.ctx = canvas.getContext('2d');
        this.state = {
            rotationX: 0,
            rotationY: 0,
            zoom: 1.0,
            isDragging: false,
            lastMouseX: 0,
            lastMouseY: 0,
            rawCoords: [], // Unscaled input
            scaledCoords: [], // After aspect ratio transform
            centerX: 0,
            centerY: 0,
            centerZ: 0,
            maxExtent: 1,
            aspectXY: 1.0,
            aspectXZ: 1.0,
            liveData: null, // Uint8Array of bytes
            bytesPerLed: 3,
            mappingStartIdx: 0,
            mappingEndIdx: 0,
            showDebug: false,
            liveDataMode: 'off' // 'off', 'device', 'group'
        };
        this._initListeners();
    }

    setLayout(coords, options) {
        const opt = options || {};
        this.state.rawCoords = coords || [];
        this.state.aspectXY = opt.aspectXY || opt.AspectXY || 1.0;
        this.state.aspectXZ = opt.aspectXZ || opt.AspectXZ || 1.0;
        this.state.mappingStartIdx = opt.mappingStartIdx || opt.MappingStartIdx || 0;
        this.state.mappingEndIdx = opt.mappingEndIdx || opt.MappingEndIdx || (this.state.rawCoords ? this.state.rawCoords.length : 0);
        this.state.showDebug = !!(opt.showDebug || opt.ShowDebug);

        this._applyScaling();
        this.render();
    }

    setLiveData(data, bytesPerLed, mode) {
        this.state.liveData = data;
        this.state.bytesPerLed = bytesPerLed || 3;
        this.state.liveDataMode = mode || 'device';
        this.render();
    }

    resetView() {
        this.state.rotationX = 0;
        this.state.rotationY = 0;
        this.state.zoom = 1.0;
        this.render();
    }

    _applyScaling() {
        /* Apply 0.5 base shrink to X/Z and aspect ratio correction */
        const ax = parseFloat(this.state.aspectXY) || 1.0;
        const az = parseFloat(this.state.aspectXZ) || 1.0;

        this.state.scaledCoords = this.state.rawCoords.map(c => {
            return {
                x: (c.x !== undefined ? c.x : c.X) * 0.5,
                y: (c.y !== undefined ? c.y : c.Y) / ax,
                z: ((c.z !== undefined ? c.z : c.Z) * 0.5) / az
            };
        });


        // Calculate bounds and center
        let minX = Infinity, maxX = -Infinity, minY = Infinity, maxY = -Infinity, minZ = Infinity, maxZ = -Infinity;
        if (this.state.scaledCoords.length === 0) {
            this.state.maxExtent = 1;
            return;
        }

        this.state.scaledCoords.forEach(c => {
            if (c.x < minX) minX = c.x; if (c.x > maxX) maxX = c.x;
            if (c.y < minY) minY = c.y; if (c.y > maxY) maxY = c.y;
            if (c.z < minZ) minZ = c.z; if (c.z > maxZ) maxZ = c.z;
        });

        // Store bounds in state for debug display
        this.state.minX = minX;
        this.state.maxX = maxX;
        this.state.minY = minY;
        this.state.maxY = maxY;
        this.state.minZ = minZ;
        this.state.maxZ = maxZ;

        this.state.centerX = (minX + maxX) / 2;
        this.state.centerY = (minY + maxY) / 2;
        this.state.centerZ = (minZ + maxZ) / 2;

        this.state.maxExtent = Math.max(maxX - minX, maxY - minY, maxZ - minZ) || 1;
    }

    _initListeners() {
        const canvas = this.canvas;
        canvas.addEventListener('mousedown', (e) => {
            this.state.isDragging = true;
            this.state.lastMouseX = e.clientX;
            this.state.lastMouseY = e.clientY;
            canvas.style.cursor = 'grabbing';
        });

        window.addEventListener('mousemove', (e) => {
            if (!this.state.isDragging) return;
            const deltaX = e.clientX - this.state.lastMouseX;
            const deltaY = e.clientY - this.state.lastMouseY;
            this.state.rotationY -= deltaX * 0.01;
            this.state.rotationX += deltaY * 0.01;
            this.state.lastMouseX = e.clientX;
            this.state.lastMouseY = e.clientY;
            this.render();
        });

        window.addEventListener('mouseup', () => {
            this.state.isDragging = false;
            canvas.style.cursor = 'grab';
        });

        canvas.addEventListener('wheel', (e) => {
            e.preventDefault();
            const delta = -e.deltaY;
            const zoomFactor = Math.pow(1.1, delta / 100);
            this.state.zoom *= zoomFactor;
            if (this.state.zoom < 0.1) this.state.zoom = 0.1;
            if (this.state.zoom > 50) this.state.zoom = 50;
            this.render();
        }, { passive: false });

        if (typeof ResizeObserver !== 'undefined') {
            new ResizeObserver(() => this.render()).observe(this.container);
        }
        canvas.style.cursor = 'grab';
    }

    render() {
        const { canvas, container, ctx, state } = this;
        if (!ctx) return;
        if (!state.scaledCoords || state.scaledCoords.length === 0) {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            return;
        }

        const dpr = window.devicePixelRatio || 1;
        const rect = container.getBoundingClientRect();
        if (rect.width <= 0 || rect.height <= 0) return;

        const targetWidth = Math.floor(rect.width * dpr);
        const targetHeight = Math.floor(rect.height * dpr);
        if (canvas.width !== targetWidth || canvas.height !== targetHeight) {
            canvas.width = targetWidth;
            canvas.height = targetHeight;
        }

        ctx.setTransform(1, 0, 0, 1, 0, 0);
        ctx.fillStyle = '#000';
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        ctx.scale(dpr, dpr);
        ctx.translate(rect.width / 2, rect.height / 2);

        const padding = 0.2;
        const usableSize = Math.min(rect.width, rect.height) * (1 - padding * 2);
        const baseScale = (usableSize * state.zoom) / state.maxExtent;

        const cosY = Math.cos(state.rotationY), sinY = Math.sin(state.rotationY);
        const cosX = Math.cos(state.rotationX), sinX = Math.sin(state.rotationX);

        const pointsWithDepth = [];
        for (let i = 0; i < state.scaledCoords.length; i++) {
            const c = state.scaledCoords[i];
            const x = c.x - state.centerX, y = c.y - state.centerY, z = c.z - state.centerZ;

            const x1 = cosY * x - sinY * z;
            const z1 = sinY * x + cosY * z;
            const y1 = cosX * y - sinX * z1;
            const z2 = sinX * y + cosX * z1;

            pointsWithDepth.push({
                px: x1 * baseScale,
                py: -y1 * baseScale, // Flip Y for canvas
                index: i,
                depth: z2
            });
        }

        // Sort by depth (Painter's algorithm)
        pointsWithDepth.sort((a, b) => a.depth - b.depth);

        pointsWithDepth.forEach(item => {
            const isActive = (item.index >= state.mappingStartIdx && item.index < state.mappingEndIdx);
            let color = "rgba(180, 180, 180, 0.4)";
            if (item.index === 0 && state.liveDataMode === 'off') color = "#add8e6";

            if (state.liveData) {
                const dataIndex = state.liveDataMode === 'group' ? item.index : (item.index - state.mappingStartIdx);
                if (dataIndex >= 0) {
                    const bpl = state.bytesPerLed;
                    const base = dataIndex * bpl;
                    if (base + 2 < state.liveData.length) {
                        const r = bpl === 4 ? state.liveData[base + 1] : state.liveData[base];
                        const g = bpl === 4 ? state.liveData[base + 2] : state.liveData[base + 1];
                        const b = bpl === 4 ? state.liveData[base + 3] : state.liveData[base + 2];
                        color = `rgb(${r},${g},${b})`;
                    }
                }
            }

            ctx.beginPath();
            ctx.arc(item.px, item.py, 3, 0, 2 * Math.PI);
            ctx.strokeStyle = (state.liveData || isActive) ? color : "rgba(100, 100, 100, 0.2)";
            ctx.lineWidth = 1.0;
            ctx.stroke();
            if (state.liveData) {
                ctx.fillStyle = color;
                ctx.fill();
            }
        });

        if (state.showDebug) {
            ctx.setTransform(dpr, 0, 0, dpr, 0, 0);
            ctx.font = "20px monospace";
            ctx.fillStyle = "#0f0";
            let y = 24;
            const lineHeight = 24;

            ctx.fillText(`Rot: ${(state.rotationX * 180 / Math.PI).toFixed(0)}°, ${(state.rotationY * 180 / Math.PI).toFixed(0)}°`, 5, y);
            y += lineHeight;
            ctx.fillText(`Zoom: ${state.zoom.toFixed(2)}x`, 5, y);
            y += lineHeight;
            ctx.fillText(`AspectXY: ${state.aspectXY.toFixed(4)}`, 5, y);
            y += lineHeight;
            ctx.fillText(`AspectXZ: ${state.aspectXZ.toFixed(4)}`, 5, y);
            y += lineHeight;
            ctx.fillText(`Coords: ${state.scaledCoords.length}`, 5, y);
            y += lineHeight;
            ctx.fillText(`Bounds X: [${state.minX.toFixed(1)}, ${state.maxX.toFixed(1)}]`, 5, y);
            y += lineHeight;
            ctx.fillText(`Bounds Y: [${state.minY.toFixed(1)}, ${state.maxY.toFixed(1)}]`, 5, y);
            y += lineHeight;
            ctx.fillText(`Bounds Z: [${state.minZ.toFixed(1)}, ${state.maxZ.toFixed(1)}]`, 5, y);
        }
    }
}
