function parseGif(gifBytes, layout) {
    // Log helper
    function log(msg) { try { console.log("[JS-GIF] " + msg); } catch (e) { } }

    // 1. Convert Data
    var u8;
    try {
        u8 = new Uint8Array(gifBytes);
        if (u8.length === 0 && gifBytes.Length > 0) throw "Conversion failed";
    } catch (e) {
        var len = gifBytes.Length;
        u8 = new Uint8Array(len);
        for (var k = 0; k < len; k++) {
            u8[k] = gifBytes[k];
        }
    }

    // 2. Decode Header
    var reader = new GifReader(u8);
    var width = reader.width;
    var height = reader.height;
    var frameCount = reader.numFrames();

    log("Dimensions: " + width + "x" + height + " Frames: " + frameCount);

    var compiled = {
        FrameCount: frameCount,
        FrameInterval: 100,
        Frames: []
    };

    // 3. Determine Layout Bounds
    // Handle C# List<Coordinate> vs JS Array
    var layoutLen = layout.length;
    if (layoutLen === undefined) layoutLen = layout.Count;
    if (layoutLen === undefined) layoutLen = 0;

    var minX = Infinity, maxX = -Infinity, minY = Infinity, maxY = -Infinity;

    if (layoutLen > 0) {
        for (var i = 0; i < layoutLen; i++) {
            var c = layout[i];
            var X = (c.X !== undefined) ? c.X : c.x;
            var Y = (c.Y !== undefined) ? c.Y : c.y;
            if (X < minX) minX = X;
            if (X > maxX) maxX = X;
            if (Y < minY) minY = Y;
            if (Y > maxY) maxY = Y;
        }
    } else {
        minX = -1; maxX = 1; minY = 0; maxY = 1;
    }

    log("Detected Bounds: X[" + minX.toFixed(2) + ", " + maxX.toFixed(2) + "] Y[" + minY.toFixed(2) + ", " + maxY.toFixed(2) + "]");

    // 4. Determine Map Bounds
    var mapMinX = minX;
    var mapMaxX = maxX;
    var mapMinY = minY;
    var mapMaxY = maxY;

    // Check if bounds fit within a reasonable "standard" box (-2..2, -1..2)
    if (minX >= -2 && maxX <= 2 && minY >= -1 && maxY <= 2) {
        log("Using Standard Twinkly Bounds (-1..1, 0..1)");
        mapMinX = -1.0;
        mapMaxX = 1.0;
        mapMinY = 0.0;
        mapMaxY = 1.0;
    } else {
        log("Using Custom Bounds (Tight Fit)");
    }

    var rangeX = mapMaxX - mapMinX;
    var rangeY = mapMaxY - mapMinY;
    if (rangeX <= 0.0001) rangeX = 1;
    if (rangeY <= 0.0001) rangeY = 1;

    // 5. Decode Frames
    var pixelBuffer = new Uint8Array(width * height * 4);

    // Get Delay
    try {
        var info = reader.frameInfo(0);
        if (info && info.delay) compiled.FrameInterval = info.delay * 10;
    } catch (e) { }

    log("Frame Interval: " + compiled.FrameInterval);

    for (var f = 0; f < frameCount; f++) {
        try {
            reader.decodeAndBlitFrameRGBA(f, pixelBuffer);
        } catch (decodeErr) {
            log("Error decoding frame " + f + ": " + decodeErr);
            continue;
        }

        var frameColors = [];

        for (var i = 0; i < layoutLen; i++) {
            var c = layout[i];
            var cX = (c.X !== undefined) ? c.X : c.x;
            var cY = (c.Y !== undefined) ? c.Y : c.y;

            // Normalize to 0..1 relative to the Map area
            var nX = (cX - mapMinX) / rangeX;
            var nY = (cY - mapMinY) / rangeY;

            // Map to Image Coords
            var x = Math.floor(nX * (width - 1));
            // Invert Y (Twinkly Y=0 is bottom, Image Y=0 is top)
            var y = Math.floor((1.0 - nY) * (height - 1));

            // Clamp
            if (x < 0) x = 0; if (x >= width) x = width - 1;
            if (y < 0) y = 0; if (y >= height) y = height - 1;

            // Index
            var idx = (y * width + x) * 4;

            var r = pixelBuffer[idx];
            var g = pixelBuffer[idx + 1];
            var b = pixelBuffer[idx + 2];

            frameColors.push({ R: r, G: g, B: b, W: 0 });
        }
        compiled.Frames.push(frameColors);
    }

    return JSON.stringify(compiled);
}
